/**
 * The PSY ATM definitions.
 * Suppose only two accounts work:
 *   - user_id = 20, pin = 4555, balance = 1234 
 *   - user_id = 30, pin = 9105, balance = 305
 */

#ifndef __ATM_H
#define __ATM_H

#include <stdbool.h>

/* Forward declaration (opaque type) */
typedef struct atm_psy atm_psy_t;

/* User's account state */
enum atm_user_state
{
    KNOWN = 0, ///< The user is known in the ATM's registry
    WRONG_PIN, ///< The user is known, but incorrect pin was supplied
    UNKNOWN,   ///< Unknown user in the ATM's registry
    BLOCKED    ///< Known user, but the account is blocked
};

/* ATM's state */
enum atm_state
{
    IDLE,         ///< The ATM Idle state, waiting for user to be authenticated
    USER_VERIFIED ///< User was verified succesfully, ready to withdraw money
};

/**
 * @brief Retrieve a newly allocated atm_psy_t struct.
 *        Only two users can authorize: 20 and 30.
 *        Use this for your testing purposes.
 *        After every test, it should be freed using `free`.
 * 
 * @return atm_psy_t* ATM's newly allocated context
 */
atm_psy_t *atm_newatm(void);

/**
 * @brief Given user_id and pin, authenticate. If the user is known, 
 *        `KNOWN` is returned and the atm's state goes to `USER_VERIFIED`.
 *        Three consecutive incorrect pin attempts result in the user becoming
 *        blocked and the function returns `BLOCKED`.
 *        If unknown user_id is supplied, `UNKNOWN` is returned.
 *        If the ATM is in the `USER_VERIFIED` state, UNKNOWN is returned.
 *
 * @param atm ATM's context
 * @param user_id The user ID
 * @param pin The user's pin
 * @return enum atm_user_state
 */
enum atm_user_state atm_authenticate(atm_psy_t *atm, int user_id, int pin);

/**
 * @brief Check's the user's account. The resulting balance is stored
 *        to the `remaining` pointer.
 * 
 * @param atm ATM's context
 * @param remaining the account's balance
 * @return true if the ATM is in the `USER_VERIFIED` state, false otherwise.
 */
bool atm_check_balance(atm_psy_t *atm, int *remaining);

/**
 * @brief Withdraw the user's money. If `how_much` > remaining balance,
 *        no money is withdrawn.
 * 
 * @param atm ATM's context
 * @param how_much how much many should be withdrawn
 * @return true if the ATM is in the `USER_VERIFIED` state and
 *         `how_much` <= remaining.
 */
bool atm_withdraw_money(atm_psy_t *atm, int how_much);

/**
 * @brief Logout the user. I.e. the ATM goes from `USER_VERIFIED`
 *        to the `IDLE` state.
 * 
 * @param atm ATM's context
 */
void atm_logout(atm_psy_t *atm);

/**
 * @brief Deallocate the instance returned by `atm_newatm`.
 * 
 * @param atm ATM's context
 */
void atm_dispose(atm_psy_t *atm);

#endif /* __ATM_H */
